/* -*-C-*-
 ##############################################################################
 #
 # File:        demo/order2.c
 # RCS:         $Id: order2.c,v 1.22 1998/07/20 22:59:27 ericb Exp $
 # Description: Host program for demonstrating order tracking with multiple
 #		E1432s
 # Created:     March 20, 1996
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that demonstrates order tracking across two E1432s.
 # It will display the time waveforms and resampled time waveforms of the 
 # first 24 channels of the E1432s.  It will print out the RPM values of 
 # both tach channels at the trigger points.
 #
 # The equipment required for this demo is:
 # 	1. VXI card cage with MXI interface to host
 #  	2. one E1432 with at least 4 input channels and a tachometer option AYD
 #	3. another E1432 with at least 4 input channels
 #
 # Set the logical address of the the E1432 with tachometer to the address
 # specified in the LA1 defined below (default 8) and set the other E1432 to
 # the address specified in the LA2 defined below (default 9).
 #
 # Hook up a sweeping signal generator as such:
 #    output set to 2 volts peak sine wave.
 #    tee output to go to channel 1 inputs of both modules as well and into
 #    both tachometer inputs on the second module.
 #
 # To run the measurement:
 #    start the program
 #    sweep the frequency between 5Hz(300 RPM) and 105Hz (6300 RPM)
 #    over 30 seconds
 #
 # The program should display the waveform in the channel 1 and 5 displays on
 # the screen.  The measurement pre-arms at 500 RPM and begins triggering and
 # updating the display at 700 RPM and every 100 RPM interval after that 
 # until 6000 rpm. The arming RPM points, RPM at trigger point of the arming 
 # tach channel, and average RPM at during the data block of other 
 # tach channel are displayed in the terminal window.
 #
 # If the trigger delay is left at -BLOCK_SIZE/2 and the trigger mode 
 # is changed to E1432_TACH_EDGE_TRIGGER, the center point of the block 
 # of data will remain stationary since we are triggering on a  tach edge 
 # and taking data +/- BLOCK_SIZE/2 on either side of that trigger.
 #
 # Note that the RPMs for the two channels are not exactly the same.  The
 # RPM of the arm/trigger channel is calculated using a quadratic fitting 
 # method across tach edge times near the trigger point, while the second
 # channel's RPM is calculated by taking the average RPM for the block.
 #
 # Altering some of the default measurement parameters below will change the
 # nature of the display.  In particular, changing the trigger mode to
 # E1432_AUTO_TRIGGER will produce a display whose center is the exact point
 # at which the RPM reached the arming RPM and won't be lined up with a zero
 # crossing of the tach.  
 #
 # Revisions:
 #
 #
 ##############################################################################
 */
#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include "e1432.h"
#include "err1432.h"
#include "xplot.h"

/*#define TRAILER*/
/*#define EXT_TRIG*/

#define NUM_MODS	2
#define LA1		8
#define LA2		9
#define NUM_CHAN_1	8
#define NUM_CHAN_2	8
#define WIDTH           140
#define HEIGHT          80
#define WARNING_MAX	100

#define RANGE		2.0
#define BLOCK_SIZE	1024
#define	RPM_INTERVAL	100.0
#define RPM_HIGH	6000
#define	RPM_LOW		700
#define	RPM_PRE_ARM	600
#define SPAN		1250
#define MAX_ORDERS	10.0
#define DELTA_ORDER	1.0
#define PRE_ARM_MODE	E1432_ARM_RPM_RUNUP
#define ARM_MODE	E1432_ARM_RPM_RUNUP
#define TRIG_MODE	E1432_AUTO_TRIGGER 
#define	RPM_MIN		120

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif


int
main(void)
{
    int  i, j, nchan1, nchan2, total_chans;
    SHORTSIZ16 status;
    LONGSIZ32 count;
    SHORTSIZ16 la[2] = {LA1, LA2};
    SHORTSIZ16 tach_list[4];
    SHORTSIZ16 total_list[36];
    SHORTSIZ16 all, inputs, tachs, tach0, tach1, arm, nonarm;
    E1432ID hw;
    float rpm[4];
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    FLOATSIZ32 *data[32];
    FLOATSIZ32 *resampled[32];
    long points = BLOCK_SIZE;
    char *plotid[64];
    int row, col, id;  
    char geometry[80];
    char title[80];
    float temp, floatRange;
    SHORTSIZ16 warning[WARNING_MAX]; 
    unsigned long warningCount;
    int checkWarnings; 
    int module, num_tachs, input_chans1, input_chans2;
    struct e1432_trailer trailer;
    SHORTSIZ16 meas_state;
    char *semabinloc = "/opt/e1432/lib/sema.bin";

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    for(i=0; i < NUM_MODS; i++)
    {
        (void) printf("Checking for E1432 with firmware at logical address %d ... ", 
							(int)la[i]);
        (void) fflush(stdout);
        error = e1432_get_hwconfig(1, &(la[i]), &cf);
        if (error)
        {
	    (void) printf("Not found.\n");
	    (void) printf("Installing firmware from %s into E1432 at la %d ...",
					       semabinloc, (int)la[i]);
	    (void) fflush(stdout);
	    error = e1432_install(1, &(la[i]), 0, semabinloc);
	    if (error)
	    {
	        (void) printf("\ne1432_install failed and returned error %s\n");
	        exit(0);
	    }
	    (void) printf("Done.\n");
        }
        else
        {
	    (void) printf("Found.\n");
        }
    }
    (void) printf("\n");
    
    CHECK(e1432_print_errors(1));

    CHECK(e1432_assign_channel_numbers(2, la, &hw));

    error = e1432_get_hwconfig(1, &la[0], &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    input_chans1 = cf.input_chans;
    nchan1 = (cf.input_chans < NUM_CHAN_1)  ? cf.input_chans : NUM_CHAN_1;
    num_tachs = cf.tach_chans;

    error = e1432_get_hwconfig(1, &la[1], &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    input_chans2 = cf.input_chans;
    nchan2 = (cf.input_chans < NUM_CHAN_2)  ? cf.input_chans : NUM_CHAN_2;
    num_tachs += cf.tach_chans;
    
    total_chans = nchan1 + nchan2;

    (void) printf("Wait for initial rpm readout before sweeping input\n");

    /* Create channel group */
    for(i=0; i < nchan1; i++)
	total_list[i] = E1432_INPUT_CHAN(i+1);
    for(; i < total_chans; i++)
	total_list[i] = E1432_INPUT_CHAN(i+1+(input_chans1 - nchan1));
    for(; i < total_chans + num_tachs; i++)
	total_list[i] = E1432_TACH_CHAN(i - total_chans + 1);

    inputs = e1432_create_channel_group(hw, total_chans, total_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group inputs returned %d\n", inputs);
	exit(0);
    }

    all = e1432_create_channel_group(hw, total_chans + num_tachs, total_list);
    if (all >= 0)
    {
	(void) printf("e1432_create_channel_group all returned %d\n", all);
	exit(0);
    }

    for(i = 0; i < num_tachs; i++)
        tach_list[i] = E1432_TACH_CHAN(i+1);

    tach0 = tach_list[0];
    tach1 = tach_list[1];
    tachs = e1432_create_channel_group(hw, num_tachs, tach_list);
    if (tachs >= 0)
    {
	(void) printf("e1432_create_channel_group for tachs returned %d\n",
		   						tachs);
	exit(0);
    }

#if 0
    CHECK(e1432_set_internal_debug(hw, all, 0x400));  
#endif

#ifdef TRAILER
    CHECK(e1432_set_append_status(hw, all, E1432_APPEND_STATUS_ON));
#endif

    /* Initialize hardware things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, RANGE));

    CHECK(e1432_set_data_size(hw, all, E1432_DATA_SIZE_32));

    CHECK(e1432_set_data_mode(hw, all, E1432_DATA_MODE_OVERLAP_BLOCK));

    row = col = 0;
    id = 0;
    temp = (float)(points - 1);

    for(i=0; i < total_chans; i++) {
        data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!data[i]) {
          (void) printf("Can't malloc data array of %d points\n", points);
          exit(0);
        }

	for(j=0; j < points; j++) 
	{
	   data[i][j] = 0.0;
	}

        resampled[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!resampled[i]) {
          (void) printf("Can't malloc resampled array of %d points\n", points);
          exit(0);
        }

	for(j=0; j < points; j++) 
	{
	   resampled[i][j] = 0.0;
	}

        CHECK(e1432_get_range(hw, total_list[i], &floatRange));

	/* standard HPUX kernal won't support enough xplot windows */
	if(id < 48)
	{
            (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * row ); 
            (void) sprintf(title, "Time %d", i + 1);

            plotid[id] = xplot_init_plot(data[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
	    if(plotid[id] == NULL)
	        (void) printf("xplot_init_plot() failed\n");

            (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
			(WIDTH + 20) * col, (HEIGHT + 40) * (row  + 1)); 
            (void) sprintf(title, "Resamp %d", i + 1);

            plotid[id+8] = xplot_init_plot(resampled[i], points, temp, 
		floatRange, -floatRange, GENERIC_TRACE, geometry, title);

	    if(plotid[id+8] == NULL)
	        (void) printf("xplot_init_plot() failed\n");

	    xplot_change_yautoscale(plotid[id], 0);
            xplot_change_yautoscale(plotid[id+8], 0);
            xplot_change_yautoscale(plotid[id+8], 0);
            xplot_set_xscale(plotid[id], (float)0.0, temp);
            xplot_set_xscale(plotid[id+8], (float)0.0, temp);
            xplot_set_yscale(plotid[id], floatRange, -floatRange);
            xplot_set_yscale(plotid[id+8], floatRange, -floatRange);
            xplot_change_xlabel(plotid[id], "Samples");
            xplot_change_xlabel(plotid[id+8], "Samples");
            xplot_change_ylabel(plotid[id], "Volts");
            xplot_change_ylabel(plotid[id+8], "Volts");
            xplot_repaint(plotid[id]); 
            xplot_repaint(plotid[id+8]); 
              
            col++;
            id++;
            if((col % 8) == 0) 
	    {
                col = 0;
                row +=2;
                id += 8;
            }
	}
    }

    arm = tach0;
    nonarm = tach1;
    
    CHECK(e1432_set_blocksize(hw, all, BLOCK_SIZE));
    CHECK(e1432_set_span(hw, all, SPAN));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_LOWER, 0.0));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_UPPER, 0.0));
    CHECK(e1432_set_active(hw, tach1, E1432_CHANNEL_ON));
    CHECK(e1432_set_pre_arm_mode(hw, all, PRE_ARM_MODE)); 
    CHECK(e1432_set_pre_arm_rpm(hw, tachs, RPM_PRE_ARM)); 
    CHECK(e1432_set_arm_mode(hw, all, ARM_MODE)); 
    CHECK(e1432_set_auto_trigger(hw, all, TRIG_MODE)); 

#ifdef EXT_TRIG
    CHECK(e1432_set_arm_channel(hw, arm, E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_channel(hw, arm, E1432_CHANNEL_OFF)); 
    CHECK(e1432_set_trigger_channel(hw, nonarm, E1432_CHANNEL_ON)); 
    CHECK(e1432_set_trigger_slope(hw, nonarm, E1432_TRIGGER_SLOPE_NEG)); 
#else
    CHECK(e1432_set_arm_channel(hw, arm, E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_channel(hw, nonarm, E1432_CHANNEL_OFF)); 
    CHECK(e1432_set_trigger_channel(hw, arm, E1432_CHANNEL_ON)); 
#endif

    CHECK(e1432_set_delta_order(hw, all, DELTA_ORDER)); 
    CHECK(e1432_set_max_order(hw, all, MAX_ORDERS)); 
    CHECK(e1432_set_rpm_low(hw, tachs, RPM_LOW)); 
    CHECK(e1432_set_rpm_high(hw, tachs, RPM_HIGH)); 
    CHECK(e1432_set_rpm_interval(hw, tachs, RPM_INTERVAL)); 

    CHECK(e1432_set_calc_data(hw, all, E1432_DATA_RESAMP_TIME)); 
    CHECK(e1432_set_decimation_output(hw, all, E1432_MULTIPASS)); 
    CHECK(e1432_set_decimation_oversample(hw, all, 
					E1432_DECIMATION_OVERSAMPLE_ON));
    CHECK(e1432_set_tach_max_time(hw, tachs, 60.0 / RPM_MIN));
    CHECK(e1432_set_trigger_master(hw, tach0, E1432_TRIGGER_MASTER_ON)); 

#if 1
    CHECK(e1432_set_trigger_delay(hw, tachs, -BLOCK_SIZE/2)); 
#else
    CHECK(e1432_set_trigger_delay(hw, tachs, 0)); 
#endif

    /* Start measurement */
    CHECK(e1432_init_measure(hw, all));

    (void)sleep(1);
    (void)printf("current rpm:	");
    for(i=0; i < num_tachs; i++)
    {
        CHECK(e1432_get_current_rpm(hw, tach_list[i], &rpm[i]));
    	(void) printf("%g		", rpm[i]);
    }
    (void)printf("\n");

    for (;;)
    {
	do  /* Wait for block available and check for errors and warnings  */
	{
	    CHECK(e1432_get_meas_state(hw, all, &meas_state));
	    if(meas_state == E1432_MEAS_STATE_TESTED)
	    {
		(void) printf("Measurement finished.\n");
		exit(0);
	    }

	    if(nchan2 > 0)
	    {
	        CHECK(e1432_send_tachs(hw, all, arm, NULL, 0, &count));
	        CHECK(e1432_send_tachs(hw, all, nonarm, NULL, 0, &count));
	    }

	    CHECK(e1432_read_register(hw, tach0, 	
					E1432_IRQ_STATUS2_REG, &status));
	    
	    if(status & E1432_IRQ_TRIGGER)
	    {
	        CHECK(e1432_send_trigger(hw, all));
	    }

	    if(status & E1432_IRQ_MEAS_ERROR)
	    {
		if(status & E1432_STATUS2_TACH_OVERFLOW)
		    (void) printf("Tach buffer overflowed in module 1\n");
		else
		    (void) printf("Fifo overflowed in module 1\n");
		exit(1);
	    }

	    /* check for warnings in second module */
	    module = 1;
	    if((checkWarnings = status & E1432_IRQ_MEAS_WARNING) == 0)
	    {
	        if(nchan2)
	        {
	            /* check for warnings in second module */
	            module = 2;
	            CHECK(e1432_read_register(hw, total_list[nchan1], 	
				    E1432_IRQ_STATUS2_REG, &status));
	            if(status & E1432_IRQ_MEAS_ERROR)
	            {
			if(status & E1432_STATUS2_TACH_OVERFLOW)
		    	    (void) printf("Tach buffer overflowed in module 2\n");
			else
		    	    (void) printf("Fifo overflowed in module 2\n");
	                exit(1);
	            }
	        }

	        checkWarnings = status & E1432_IRQ_MEAS_WARNING;
	    }

	    if(checkWarnings)
	    {
		/* read out all measurement warnings */
		do
		{
		    CHECK(e1432_get_meas_warning(hw, all, 
						 warning, WARNING_MAX,
						 &warningCount));
	   
		    if(warningCount)
		    {
	       		(void) printf("Module %d: %d Warning", 
							module, warningCount);
			if(warningCount > 1) 
			    (void) printf("s");
			(void) printf(":\n");
		    }

           	    for(i=0; i < warningCount; i++)
	   	    {
			(void) printf("    %s\n",
				      e1432_get_warning_string(warning[i]));
	   	    }

		} while (warningCount > 0);
	    }

	}while(e1432_block_available(hw, all)  == 0); 

        if (status < 0) {
	    (void) printf("status error %d\n", status);
	    exit(0);
        }

#ifndef TRAILER
	for(i = 0; i < num_tachs; i++)
	    CHECK(e1432_get_data_rpm(hw, tach_list[i], &rpm[i]));
#endif

	for(i=0; i < total_chans; i++) {	/* read time data */
            error = e1432_read_float32_data(hw, total_list[i],
				E1432_TIME_DATA, data[i], points, 
				&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
              exit(0);
            }

#ifdef TRAILER 
	    if(i==0) 	/* get tachs from first module */
	    {
		rpm[0] = trailer.rpm1;
		rpm[1] = trailer.rpm2;
	    }
	    
	    if(i == nchan1)	/* get tachs from second module */
	    {
		rpm[2] = trailer.rpm1;
		rpm[3] = trailer.rpm2;
	    }
#endif

	}

	id = 0;
        for(i=0; i < total_chans ; i++) {
            error = e1432_read_float32_data(hw, total_list[i], 
			E1432_RESAMP_DATA, resampled[i], points, 
							&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
              exit(0);
            }
            if (count != points) {
              (void) printf("Actual count was %d\n", count);
              exit(0);
            }

	    /* standard HPUX kernal won't support enough xplot windows */
            if(id < 48)
            {
                xplot_check_events(plotid[id]);
                xplot_data_update(plotid[id]);
                xplot_check_events(plotid[id+8]);
                xplot_data_update(plotid[id+8]);
            }

	    if((++id%8)==0) 
	        id += 8;
        }

        (void) printf("rpm =  ");
	for(j = 0; j < num_tachs; j++)
            (void) printf("%g		", rpm[j]);
	(void)printf("\n");

    }
    /*NOTREACHED*/
    return 0;
}

